/*
 * SoulOrb.h
 *
 * Created 9/7/2009 By Johnny Huynh
 *
 * Version 00.00.02 9/12/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef SOUL_ORB_H
 #define SOUL_ORB_H
 
 template <typename T> class SoulOrb;
 
 #include "CollisionHandlerEventNode.h"
 #include "ProjectileDamageObject.h"
 
 #include "global.h"
 
 // For applying effects
 #include "EffectInfo.h"
 #include "PushBackEffectInfo.h"
 #include "InterruptEffectInfo.h"
 
 #define SOUL_ORB_COLLIDER_NAME "SoulOrb Collider"
 
 /**
  * Class specification for SoulOrb
  * SoulOrb is a projectile that deals damage
  */
 template <typename T>
 class SoulOrb : public ProjectileDamageObject<T>
 {
 // Data Members
 private:
 
 // Local Functions
 public:
    SoulOrb( const std::string& name, const T& damage = ZERO ); 
    SoulOrb( const std::string& name, const T& damage, const EffectInfoCollection<T>& effect_info_collection ); 
    SoulOrb( const SoulOrb<T>& soul_orb );
    virtual ~SoulOrb();
    inline SoulOrb<T>& operator=( const SoulOrb<T>& soul_orb );
    
    // overloaded functions (NodePath)
    /*static void init_type() {
                                std::string template_type( typeid( T ).name() );
                                register_type(_type_handle, "SoulOrb<" + template_type + ">" );
                            }*/
 
 // Private Functions
 private:
    inline void initialize();
    
 // Public Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 SoulOrb<T>::SoulOrb( const std::string& name, const T& damage )
            : ProjectileDamageObject<T>( name, damage ),
              DamageObject<T>( name, damage )
 {
    SoulOrb<T>::initialize();
 }
 
 /**
  * Alternative Constructor
  */
 template <typename T>
 SoulOrb<T>::SoulOrb( const std::string& name, const T& damage, const EffectInfoCollection<T>& effect_info_collection )
            : ProjectileDamageObject<T>( name, damage, effect_info_collection ),
              DamageObject<T>( name, damage, effect_info_collection )
 {
    SoulOrb<T>::initialize();
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 SoulOrb<T>::SoulOrb( const SoulOrb<T>& soul_orb )
            : ProjectileDamageObject<T>( soul_orb ),
              DamageObject<T>( soul_orb )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 SoulOrb<T>::~SoulOrb()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified SoulOrb to this SoulOrb.
  *
  * @param (const SoulOrb<T>& soul_orb )
  * @return SoulOrb<T>&
  */
 template <typename T>
 inline SoulOrb<T>& SoulOrb<T>::operator=( const SoulOrb<T>&  soul_orb )
 {
    ProjectileDamageObject<T>::operator=( static_cast< ProjectileDamageObject<T> >( soul_orb ) );
    
    return *this;
 }
 
 /** PRIVATE FUNCTIONS **/
 
 /**
  * initialize() sets up this Soul Orb.
  */
 template <typename T>
 inline void SoulOrb<T>::initialize()
 {
    DamageObject<T>::add_effect( new PushBackEffectInfo<T>( 0.15, 1.0f, 0.15 ) );
    DamageObject<T>::add_effect( new InterruptEffectInfo<T>() );
    
    PT(CollisionHandlerEventNode<T>) collider_Ptr( new CollisionHandlerEventNode<T>( SOUL_ORB_COLLIDER_NAME ) );
    DamageObject<T>::add_collider( collider_Ptr );
    collider_Ptr->collision_node()->add_solid( new CollisionSphere( ZERO, ZERO, ZERO, 1.0f ) );
    collider_Ptr->show();
    collider_Ptr->turn_off_from_collision(); // make the collider only detect collisions when another 
                                             // collider collides into this collider
    
    // re-position the collider
    collider_Ptr->set_z( 1.0f );
    
    // Event Handling
    collider_Ptr->add_in_pattern();
    collider_Ptr->add_event(); // may want to change to handle_event and automatically called when adding in patterns
    
    collider_Ptr->reparent_to( *this ); // attach the collider to the SoulOrb
 }
 
 #undef SOUL_ORB_COLLIDER_NAME
 
 #endif // SOUL_ORB_H